<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\User;

use Exception;
use Erlage\Photogram\Settings;
use Erlage\Photogram\SystemLogger;
use Erlage\Photogram\Tools\Crypto;
use Erlage\Photogram\Tools\Fso\Storage;
use Erlage\Photogram\Tools\Fso\ImageUploader;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Data\Models\User\UserEnum;
use Erlage\Photogram\Data\Tables\Sys\RestTable;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Models\User\UserBuilder;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Dtos\Common\DisplayItemDTO;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;
use Erlage\Photogram\Data\Dtos\User\UserDisplayImageDTO;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowEnum;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowFinder;

final class UserFieldEditor extends ExceptionalRequests
{
    public static function editUserField(string $attribute): void
    {
        self::process(function () use ($attribute)
        {
            self::internalFieldEditor($attribute);
        });
    }

    public static function internalFieldEditor($attribute): void
    {
        /*
        |--------------------------------------------------------------------------
        | get data from request
        |--------------------------------------------------------------------------
        */

        $fieldValueFromReq = self::$request -> findKey($attribute, RequestTable::PAYLOAD, UserTable::TABLE_NAME);

        /*
        |--------------------------------------------------------------------------
        | make sure user is authenticated
        |--------------------------------------------------------------------------
        */

        self::userEnsureAuthenticated();

        /*
        |--------------------------------------------------------------------------
        | if a rich attribute, make sure to deserialize it as findKey returns string
        | always
        |--------------------------------------------------------------------------
        */

        $deSerializers = UserTable::deSerializers();

        if (isset($deSerializers[$attribute]))
        {
            $fieldValueToSet = $deSerializers[$attribute]($fieldValueFromReq);
        }
        else
        {
            $fieldValueToSet = $fieldValueFromReq;
        }

        /*
        |--------------------------------------------------------------------------
        | ensure value is changed
        |--------------------------------------------------------------------------
        */

        $getter = 'get' . \ucfirst(UserTable::CAMEL_CASE_MAP[$attribute]);

        if ($fieldValueToSet == self::$authedUserModel -> {$getter}())
        {
            self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getDataMap());

            return;
        }

        /*
        |--------------------------------------------------------------------------
        | if changing account privacy setting
        |--------------------------------------------------------------------------
        */

        if (UserTable::META_IS_PRIVATE == $attribute)
        {
            $isChangingToPublicAccount = self::$authedUserModel -> isPrivate() && $fieldValueToSet == UserEnum::META_IS_PRIVATE_NO;
        }
        elseif (UserTable::EMAIL == $attribute)
        {
            /*
            |--------------------------------------------------------------------------
            | if updating email, make sure to mark new email as not-verified
            |--------------------------------------------------------------------------
            */

            if (self::$authedUserModel -> getEmail() != $fieldValueToSet)
            {
                self::$authedUserModel -> update(
                    array(
                        UserTable::META_IS_EMAIL_VERIFIED => UserEnum::META_IS_EMAIL_VERIFIED_NO,
                    )
                );
            }
        }

        /*
        |--------------------------------------------------------------------------
        | try updating data
        |--------------------------------------------------------------------------
        */

        self::$authedUserModel -> update(
            array(
                $attribute => $fieldValueToSet,
            )
        );

        self::$authedUserModel -> save();

        /*
        |--------------------------------------------------------------------------
        | accept all pending requests if account is changing to public mode
        |--------------------------------------------------------------------------
        */

        if (isset($isChangingToPublicAccount))
        {
            $finder = (new UserFollowFinder())
                -> setFollowedUserId(self::$authedUserModel -> getId())
                -> setMetaIsPending(UserFollowEnum::META_IS_PENDING_YES)
                -> find();

            if ($finder -> isFound())
            {
                while ($finder -> canPop())
                {
                    $followModel = $finder -> popModelFromResults();

                    $followModel -> update(
                        array(
                            UserFollowTable::META_IS_PENDING, UserFollowEnum::META_IS_PENDING_NO,
                        )
                    );

                    $followModel -> save();
                }
            }
        }

        /*
        |--------------------------------------------------------------------------
        | return updated user' map
        |--------------------------------------------------------------------------
        */

        self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getDataMap());
    }

    public static function updateUserPassword(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get fields
            |--------------------------------------------------------------------------
            */

            $oldPasswordFromReq = self::$request -> findKey(
                UserTable::PASSWORD,
                RequestTable::PAYLOAD,
                UserTable::TABLE_NAME
            );

            $newPasswordFromReq = self::$request -> findKey(
                UserTable::EXTRA_NEW_PASSWORD,
                RequestTable::PAYLOAD,
                UserTable::TABLE_NAME
            );

            $retypeNewPasswordFromReq = self::$request -> findKey(
                UserTable::EXTRA_RETYPE_NEW_PASSWORD,
                RequestTable::PAYLOAD,
                UserTable::TABLE_NAME
            );

            self::ensureValue(
                ResponseConstants::ERROR_BAD_REQUEST_MSG,
                $oldPasswordFromReq,
                $newPasswordFromReq,
                $retypeNewPasswordFromReq
            );

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | verify the provided password
            |--------------------------------------------------------------------------
            */

            if (
                ! Crypto::verifyClearPasswordWithDatabaseHash(
                    $oldPasswordFromReq,
                    self::$authedUserModel -> getPassword()
                )
            ) {
                throw new RequestException(ResponseConstants::D_ERROR_USER_NOT_MATCHED_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | check whether new & retype password are different
            |--------------------------------------------------------------------------
            */

            if ($newPasswordFromReq !== $retypeNewPasswordFromReq)
            {
                throw new RequestException(ResponseConstants::D_ERROR_USER_PASSWORD_MISMATCH_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | update user's fields
            |--------------------------------------------------------------------------
            */

            self::$authedUserModel -> update(
                array(
                    UserTable::PASSWORD    => $newPasswordFromReq,
                    UserTable::META_ACCESS => '',
                )
            );

            /*
            |--------------------------------------------------------------------------
            | save
            |--------------------------------------------------------------------------
            */

            self::$authedUserModel -> save();

            /*
            |--------------------------------------------------------------------------
            | return updated user' map
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getDataMap());
        });
    }

    public static function removeUserProfilePicture(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | object references for delete
            |--------------------------------------------------------------------------
            */

            $obsoluteDisplayImageDTO = self::$authedUserModel -> getDisplayImage();

            /*
            |--------------------------------------------------------------------------
            | update user's fields
            |--------------------------------------------------------------------------
            */

            self::$authedUserModel -> update(
                array(
                    UserTable::DISPLAY_IMAGE => UserBuilder::defaultValues()[UserTable::DISPLAY_IMAGE],
                )
            );

            /*
            |--------------------------------------------------------------------------
            | save
            |--------------------------------------------------------------------------
            */

            self::$authedUserModel -> save();

            /*
            |--------------------------------------------------------------------------
            | clean objects from storage
            |--------------------------------------------------------------------------
            */

            Storage::disk($obsoluteDisplayImageDTO -> getHost())
                -> deleteDTO($obsoluteDisplayImageDTO);

            /*
            |--------------------------------------------------------------------------
            | return updated user' map
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getDataMap());
        });
    }

    public static function uploadUserProfilePicture(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $newDisplayImageInfo = self::$request -> getCollection(UserTable::DISPLAY_IMAGE, RestTable::ATTACHEMENTS, UserTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $newDisplayImageInfo);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | create image uploader
            |--------------------------------------------------------------------------
            */

            $host = Settings::getString(ServerConstants::SS_ENUM_STORAGE_DISK_USER_IMAGES);
            $resolution = Settings::getInt(ServerConstants::SS_INT_COMPRESS_USER_DISPLAY_IMAGE_FILE_RES);
            $compressedQuality = Settings::getInt(ServerConstants::SS_INT_COMPRESS_USER_DISPLAY_IMAGE_FILE_QUALITY);
            $maxFileSizeInMegaBytes = Settings::getInt(SystemConstants::SETTING_MAX_USER_DISPLAY_IMAGE_FILE_SIZE);
            $supportedExtensions = Settings::getString(SystemConstants::SETTING_SUPPORTED_USER_DISPLAY_IMAGE_FILE_FORMAT);

            $imageUploader = (new ImageUploader())
                -> setHost($host)
                -> setUserId(self::$authedUserModel -> getId())
                -> setFilespace(DisplayItemDTO::FILESPACE_USER)

                -> setSaveResolution($resolution)
                -> setCompressedQuality($compressedQuality)
                -> setMaxSizeInMegaBytes($maxFileSizeInMegaBytes)
                -> setSupportedExtensions($supportedExtensions)

                -> setTemporaryFile($newDisplayImageInfo)

                -> prepare();

            if ( ! $imageUploader -> isValid() || $imageUploader -> isNotSupported())
            {
                throw new RequestException(ResponseConstants::D_ERROR_USER_DISPLAY_IMAGE_FILE_FORMAT_MSG);
            }

            if ($imageUploader -> isOverSized())
            {
                throw new RequestException(ResponseConstants::D_ERROR_USER_DISPLAY_IMAGE_FILE_SIZE_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | finish upload
            |---------------------------------------------------------------------------
            */

            $uploadHandle = $imageUploader -> process();

            // check if image upload has succeeded

            if ( ! $uploadHandle -> processed)
            {
                // upload failed, log internally

                SystemLogger::internalException(new Exception($uploadHandle -> error));

                // something went wrong

                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            // else get DTO object

            $displayImageDTO = $imageUploader -> getDisplayItemDTO();

            /*
            |--------------------------------------------------------------------------
            | object references for delete
            |--------------------------------------------------------------------------
            */

            $obsoluteDisplayImageDTO = self::$authedUserModel -> getDisplayImage();

            /*
            |--------------------------------------------------------------------------
            | update user's fields
            |--------------------------------------------------------------------------
            */

            self::$authedUserModel -> update(
                array(
                    UserTable::DISPLAY_IMAGE => (new UserDisplayImageDTO())
                        -> setHost($displayImageDTO -> getHost())
                        -> setIdentifier($displayImageDTO -> getIdentifier()),
                )
            );

            /*
            |--------------------------------------------------------------------------
            | save
            |--------------------------------------------------------------------------
            */

            self::$authedUserModel -> save();

            /*
            |--------------------------------------------------------------------------
            | clean objects from storage
            |--------------------------------------------------------------------------
            */

            Storage::disk($obsoluteDisplayImageDTO -> getHost())
                -> deleteDTO($obsoluteDisplayImageDTO);

            /*
            |--------------------------------------------------------------------------
            | return updated user' map
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getDataMap());
        });
    }
}
